#!/bin/bash
#
# $Id: $
#
# Copyright (c) 2013, Juniper Networks, Inc.
# All rights reserved.
#
# Junos VM installation script called by jinstall-qfx-5x00 img package
# to install the Junos VM on hypervisor.
#
# Usage: install-vm [add|rollback] <image>
#
. /etc/rc.d/init.d/functions
. /etc/gen_functions.sh
. /sbin/gen_install_functions.sh


# Enable for debug
# set -x

inst_dir=/var/tmp/preinstall
upgrade_dir=/var/etc/upgrade

op=
image=
opt_slot=
opt_verbose=0

usage() {
    echo "Usage: $0 [add|rollback|compat-check] [-s 0:1] [-f <force>] [-v <verbose>] <pkg>"
    exit 1
}

while true; do
    case "$1" in
    add|rollback|compat-check)
        op=$1; shift ;;

    -f|--force)
        force_option="-f"; shift ;;

    -s|--slot)
        [ "$2" != 0 -a "$2" != 1 ] && usage
        opt_slot=$2; shift 2 ;;

    -v|--verbose)
        opt_verbose=1; shift ;;

    -*)
        usage ;;

    *)
        [ -z $1 ] && break
        vj_pkg=$1; shift
    esac
done

trace() {
  local level=$1
  shift

  case "${level}" in
  "info")  echo "$@" ;;
  "error") echo "$@" ;;
  "debug") [ $opt_verbose == 1 ] && echo "$@" ;;
  esac

  logger "$@"
}   

error() {
  if [ ! -z "$1" ]; then
    logger "ERROR: $@"
  fi
  exit 1
}

version_max() {
  for _v in "$@"; do echo $_v; 
  done | sort -t "."  +0 -1 -n +1 -2 -n +2 -3 -n | tail -1
}

version_lt() {
  m=`version_max "$@"`
  test $m = $2 && $m != $1
}

version_ge() {
  m=`version_max "$@"`
  test $m = $1
}

# All the dotted numbers before .0.XXX 
# are compared against each other
# The input parameters are $1 and $2
# in the cvs revision format
# eg: is_version_compatible "1.1.2.1" "1.1.2.0.3"
is_version_compatible()
{
  local x="${1%.0*}"
  local y="${2%.0*}"

  [ "$x" = "$y" ]  && return 0
  return 1
}

cleanup_and_exit_on_error()
{
  trace info "ERROR: encountered during install."
  trace info "Rolling back and cleaning up the installation."
  rollback_package

  [ -d $inst_dir ]  && rm -rf $inst_dir
  [ -d $upgrade_dir ]  && rm -rf $upgrade_dir
  [ -f $vj_pkg ]  && rm -rf $vj_pkg


  exit 1
}


#
# cleanup_img <img_name>
#
#
cleanup_img() {
  local img_name=$1
  local real_pkg recover_pkg

  trace info "Deleting $img_name"
  [ ! -f $img_name ] && return

  # delete the pkg in /spare
  real_pkg=`readlink $img_name`
  trace debug "Deleting pkg $real_pkg"
  rm -f $img_name
  [ -f $real_pkg ] && rm -f $real_pkg

  # delete recovery pkg
  recover_pkg="/recovery/junos/`basename $real_pkg`"
  if [ -f $recover_pkg ]; then
    trace debug "Deleting recovery image $recover_pkg"
    rm -f $recover_pkg
  else
    # need to strip out the install timestamp from pkg name
    recover_pkg=`echo $recover_pkg | sed "s/-[0-9]*.img$/img/"`
    trace debug "Deleting recovery image $recover_pkg"
    [ -f $recover_pkg ] && rm -f $recover_pkg
  fi
}

rename_pkg() {
  local from=$1
  local to=$2

  if [ -f $from ]; then
    mv -f $from $to
    sync
  fi
}

extract_file() {
  local pkg=$1
  local dst_path=`dirname "$2"`
  local file=`basename "$2"`

  if [ -f $1 ]; then
    tar -C $dst_path -xzf $pkg $file || { cleanup_and_exit_on_error; }
    sync
  else
    trace error "package: $pkg is not found, extract abort"
    cleanup_and_exit_on_error
  fi
}

install_img() {
  local pkg=$1
  local instance=$2
  local pkg_name=`basename $pkg .img.gz`
  local install_time=`date +'%Y%m%d%H%M'`
  local dst_path=/recovery/junos/${pkg_name}-${install_time}.img

  local prev_image=""
  if [ -L /recovery/junos/previous_junos_image ]; then
    prev_image=`readlink /recovery/junos/previous_junos_image`
    [ -z $prev_image ] && { trace error "Could not find the image pointed by /recovery/junos/previous_junos_image"; cleanup_and_exit_on_error; }
    [ -f $prev_image ] || { trace error "Could not find the image pointed by /recovery/junos/previous_junos_image"; cleanup_and_exit_on_error; }
  fi

  local current_image=""
  if [ -L /recovery/junos/current_junos_image ]; then
    current_image=`readlink /recovery/junos/current_junos_image`
    [ -z $current_image ] && { trace error "Could not find the image pointed by /recovery/junos/current_junos_image"; cleanup_and_exit_on_error; }
    [ -f $current_image ] || { trace error "Could not find the image pointed by /recovery/junos/current_junos_image"; cleanup_and_exit_on_error; }
  else
    trace error "Could not find /recovery/junos/current_junos_image"      
    cleanup_and_exit_on_error
  fi

  # Critical section : 
  # If power goes or junos reboot at this point
  # Things can go really bad unless cleaned up. 
  # vehostd should clean up if it sees a pending file flag with upgrade: pending.
  echo "New Active Junos Image: $dst_path" > /var/run/vjunos-upgrade-pending
  echo "prev_image: $prev_image" >> /var/run/vjunos-upgrade-pending
  echo "current_image: $current_image" >> /var/run/vjunos-upgrade-pending
  echo "upgrade: pending" >> /var/run/vjunos-upgrade-pending

  # backup the current soft links
  backup_soft_links

  if [ -z "$prev_image" ]; then
    ln -sf $current_image /recovery/junos/previous_junos_image || { restore_soft_links; cleanup_and_exit_on_error; }
  else
    # Delete any old images, if they exist.
    local old_image=""
    if [ -L /recovery/junos/old_junos_image ]; then
      old_image=`readlink /recovery/junos/old_junos_image`
      # Delete the old image.
      rm -f ${old_image}
      # Delete the symlink to the old image.
      rm -f /recovery/junos/old_junos_image
    fi

    # Change previous_junos_image link to point to the image pointed by current_junos_image
    # Change current_junos_image link to point  to the new image
    rm -f /recovery/junos/previous_junos_image ${prev_image}
    ln -sf $current_image /recovery/junos/previous_junos_image  || { restore_soft_links; cleanup_and_exit_on_error; }     
  fi

  #
  # Extract image to /recovery/junos
  #
  trace info  "Extracting $pkg_name ..."
  trace debug "Extracting $pkg to $dst_path"
  [ ! -f "$pkg" ] && { trace error "Image $pkg not found"; restore_soft_links; cleanup_and_exit_on_error; }
  sync
  gunzip -c $pkg > $dst_path || { restore_soft_links; cleanup_and_exit_on_error; }
  sync


  # Associate latest image as current image
  latest_image=/recovery/junos/${pkg_name}-${install_time}.img
  rm -f /recovery/junos/current_junos_image
  ln -sf $latest_image /recovery/junos/current_junos_image || { restore_soft_links; cleanup_and_exit_on_error; }

  #
  # Switch new VM image
  #
  trace debug "Switching active Junos to $latest_image"
  switch_image_to_current $instance

  # Purposefully clearing : to differentiate half baked upgrade vs finished
  # upgrade steps.
  echo "New Active Junos Image: $dst_path" > /var/run/vjunos-upgrade-pending
  echo "upgrade: completed" >> /var/run/vjunos-upgrade-pending
  trace info "Install $pkg_name completed"
  sync
  return 0
}

backup_soft_links()
{
  rm -rf /var/tmp/linkbackup
  mkdir -p /var/tmp/linkbackup
  cp -a /recovery/junos/previous_junos_image /var/tmp/linkbackup/. > /dev/null 2>&1
  cp -a /recovery/junos/current_junos_image /var/tmp/linkbackup/. > /dev/null 2>&1
}

restore_soft_links()
{
  cp -a /var/tmp/linkbackup/* /recovery/junos/
  rm -rf /var/tmp/linkbackup
}

rollback_img() {
  local instance=$1

  trace info "Rollback Junos Image..."

  if [ ! -f /var/run/vjunos-upgrade-pending ]; then
    trace debug "Rollback nothing to do"
    return 0
  fi

  # Change previous_junos_image link to point to the image pointed by current_junos_image
  # Change current_junos_image link to point  to the new image

  #Delete any instances of the old image and it's symlink.
  if [ -L /recovery/junos/old_junos_image ]; then
    old_image=`readlink /recovery/junos/old_junos_image`
    rm -f ${old_image}
    rm -f /recovery/junos/old_junos_image
  fi

  prev_image=""
  if [ -L /recovery/junos/previous_junos_image ]; then
    prev_image=`readlink /recovery/junos/previous_junos_image`
    if [ $? -ne 0 ]; then
      trace error "Could not find image pointed by /recovery/junos/previous_junos_image"
      cleanup_and_exit_on_error
    fi
  fi

  current_image=""
  if [ -L /recovery/junos/current_junos_image ]; then
    current_image=`readlink /recovery/junos/current_junos_image`
    if [ $? -ne 0 ]; then
      trace error "Could not find image pointed by /recovery/junos/current_junos_image"
      cleanup_and_exit_on_error
    else
      rm -f ${current_image}
    fi
  else
    trace error "Could not find /recovery/junos/current_junos_image"      
    cleanup_and_exit_on_error
  fi

  # Critical section : 
  backup_soft_links

  # Create a backup of current soft links
  rm -f /recovery/junos/previous_junos_image
  rm -f /recovery/junos/current_junos_image

  if [ ! -z "$prev_image" ]; then
    ln -sf $prev_image /recovery/junos/current_junos_image || { restore_soft_links; cleanup_and_exit_on_error; }
  else
    trace error "Could not find prev_image"  
    restore_soft_links
    cleanup_and_exit_on_error
  fi

  #
  # Switch prev VM image
  #
  trace debug "Rollingback Junos to $prev_image"
  switch_image_to_current $instance

  # cleanup the pending file
  rm -f /var/run/vjunos-upgrade-pending
  trace info "Rollback Done"
  sync
  return 0
}

switch_image_to_current()
{
  local current_junos_img=`readlink /recovery/junos/current_junos_image`
  local instance=${1:-`get_running_vjunos_instance /junos`}

  if [  -z  $instance ]; then
    trace error "Could not find junos intance identifier"   
    cleanup_and_exit_on_error
  fi
  rm -f /junos/images/$instance/vjunos.img
  /usr/bin/qemu-img create -b $current_junos_img -f qcow2 \
     /junos/images/$instance/vjunos.img > /dev/null 2>&1 || { cleanup_and_exit_on_error; }
  sync
}

stage_vjunos_img() {

  local pkg=$1
  local pkg_name=`basename $pkg .img.gz`
  local install_time=`date +'%Y%m%d%H%M'`
  local dst_path=/${upgrade_dir}/${pkg_name}-${install_time}.img

  #
  # Extract image
  #
  mkdir -p ${upgrade_dir}
  trace info "Extracting image..."
  trace debug "Extracting $pkg to $dst_path"
  [ ! -f "$pkg" ] && { trace error "Image $pkg not found"; cleanup_and_exit_on_error; }
  sync
  gunzip -c $pkg > $dst_path || { cleanup_and_exit_on_error; }
  sync

}

install_hypervisor_pkg() {

  local pkg=$1
  local vimg_pkg_gz=$2
  local upgrade_flags="upgrade-install prev-version=${cur_ver}"

  trace info "Upgrade Host OS to $hyp_ver..."
  trace debug "Upgrade Host from $cur_hyp_ver to $hyp_ver..."
  if [ -f /var/run/upgrade-reboot-pending -a -d /boot/boot/backup ]; then
    trace error "ERROR: There is a pending installation please roll back and try"      
    cleanup_and_exit_on_error
  else

    # extract hypervisor os files
    cd $inst_dir
    tar --exclude jinstall-*.img.gz -xzf $pkg || { cleanup_and_exit_on_error; }
    sync

    cd /boot/boot
    rm -fr backup || { cleanup_and_exit_on_error; }
    mkdir -p backup || { cleanup_and_exit_on_error; }
    mv System.map-* backup/. || { cleanup_and_exit_on_error; }
    mv config-* backup/. || { cleanup_and_exit_on_error; }
    mv initr*-* backup/. || { cleanup_and_exit_on_error; }
    mv symvers-* backup/. || { cleanup_and_exit_on_error; }
    mv vmlinuz-* backup/. || { cleanup_and_exit_on_error; }
    mv *.sum     backup/ || { cleanup_and_exit_on_error; }
    cp -a grub/grub.conf backup/. || { cleanup_and_exit_on_error; }
    cp -a /etc/jrelease backup/. || { cleanup_and_exit_on_error; }

    # install the new os files
    cp -a $inst_dir/System.map-* . || { cleanup_and_exit_on_error; }
    cp -a $inst_dir/config-* . || { cleanup_and_exit_on_error; }
    cp -a $inst_dir/initr*-* . || { cleanup_and_exit_on_error; }
    cp -a $inst_dir/symvers-* . || { cleanup_and_exit_on_error; }
    cp -a $inst_dir/vmlinuz-* . || { cleanup_and_exit_on_error; }

    # copy the md5sum data
    cat $inst_dir/contents.md5sum | grep 'vmlinuz\|initrd' > linux-md5check.sum
    cat $inst_dir/contents.md5sum | grep 'jinstall-' > junos-md5check.sum

    # create new grub.conf
    vmlinuz=`ls | grep vmlinuz-`
    initrd=`ls | grep initrd-`
    if [ -z "$initrd" ];then
      initrd=`ls | grep initramfs-`
    fi

    # update the kernel and ramdisk options. Also change the title to indicate
    # its an upgrade process
    cat backup/grub.conf | sed -n '/Debug/q;p' | sed  '/title/c title Running Host Upgrade [Do not power cycle!!]' | sed -e "s/vmlinuz-.*.x86_64/$vmlinuz/" -e "s/initramfs-.*.img/$initrd/" > grub/grubtemp1.conf

    cat grub/grubtemp1.conf | sed -e "s/initrd-.*.img/$initrd/" > grub/grubtemp.conf

    # Add kernel command line option to inidicate upgrade
    cat grub/grubtemp.conf | \
    sed -e '/vmlinuz/s|$| '"${upgrade_flags}"'|' > grub/grub.conf
    rm grub/grubtemp.conf
    rm grub/grubtemp1.conf

    # verify sanity and stage the junos image
    cp junos-md5check.sum $inst_dir/
    cd $inst_dir
    md5sum -c junos-md5check.sum
    cd -

    stage_vjunos_img $vimg_pkg_gz
    sync

    echo "Upgrade OS to $hyp_ver" > /var/run/upgrade-reboot-pending
    sync
    trace info "Host upgrade staging completed. Need reboot to complete upgrade installation."
  fi
  return 0
} 

rollback_hypervisor_pkg() {

  if [ -f /var/run/upgrade-reboot-pending -a -d /boot/boot/backup ]; then
    local ver=`cat /boot/boot/backup/jrelease`

    trace info "Rollback Host OS to $ver..."

    cd /boot/boot

    if [ -f backup/grub.conf ]; then
      rm -f System.map-*
      rm -f config-*
      rm -f initr*-*
      rm -f symvers-*
      rm -f vmlinuz-*
      mv backup/* .
      rm -fr backup
      rm -fr jrelease
      mv grub.conf grub/.
      sync
    fi  

    rm -fr ${upgrade_dir}
    rm -f /var/run/upgrade-reboot-pending
    sync
    trace info "Rollback Host OS done"
  else
    trace debug "Nothing to do, done"
  fi

  return 0
}

# Note :
# On QFX insyde BIOS , the next boot value
# for SSD0 is 4 and SSD1 is 5.
#
set_boot_from_same_disk() 
{
  local boot_id=$(getbootarg boot0=)
  if [ -z $boot_id ]; then
    printf "WARNING: Changing next boot to SSD1 \n" 
    bios -w 1 5
  else
    printf "WARNING: Changing next boot to SSD0 \n" 
    bios -w 1 4
  fi
  local new_mask=0
  local current_mask=`bios -r 0`
  # Force enable SSD0 and SSD1 as this requires image to be 
  # booted from SATA disks
  let "new_mask=0x18 | $current_mask"
  printf -v new_mask "0x%x" "$new_mask"
  printf "WARNING: Changing boot device enable from  0x%02X to 0x%02X\n" \
  "$current_mask" "$new_mask"
  bios -w 0 $new_mask
}

install_package() {
  local pkgfile=$1
  local instance=$2
  local vimg_pkg_gz=`tar tfz $pkgfile | grep ".img.gz$"`

  rm -fr $inst_dir
  mkdir -p $inst_dir
  
  # Extract the package
  extract_file $pkgfile $inst_dir/$vimg_pkg_gz

  # extract metatag file
  extract_file $pkgfile $inst_dir/metatags
  min_ver=`cat $inst_dir/metatags | grep 'MINIMUM-VERSION' | sed 's/^.*=//'`
  hyp_ver=`cat $inst_dir/metatags | grep 'HYPERVISOR-RELEASE' | sed 's/^.*=//'`
  # for legacy compatibility
  cur_ver=`cat /etc/jrelease` 
  cur_hyp_ver=`cat /etc/jrelease` 
  # current approach
  [ -f /etc/compatible_version ] && cur_ver=`cat /etc/compatible_version`

  trace debug "Force option: $force_option"

  # Check whether Host OS upgrade is needed?
  # if force option is not specified and if current version is newer
  # than required min version , then upgrade only the guest vm image
  if is_version_compatible $cur_ver $min_ver  && [ "$force_option" != "-f" ]; then
    trace debug "Current compat version: $cur_ver installed compat version: $min_ver"
    trace debug "Current host version: $cur_hyp_ver is compatible with installed version: $hyp_ver"
    trace debug "The new guest will be compatible with current host software."
    trace debug "Host upgrade is not required."
    # Install the vjunos.
    if [ -f $inst_dir/$vimg_pkg_gz ]; then
      install_img $inst_dir/$vimg_pkg_gz $instance
    else
      trace error "Extract $vimg_pkg_gz failed, install abort!"
      cleanup_and_exit_on_error
    fi
  else
    if [ "$force_option" == "-f" ]; then
      trace debug "Host upgrade is FORCED"
      trace debug "Current compat version: $cur_ver installed compat version: $min_ver"
      trace debug "Current host version: $cur_hyp_ver"
      trace debug "New host version: $hyp_ver"
    else
      trace debug "Current compat version: $cur_ver installed compat version: $min_ver"
      trace debug "Current host version: $cur_hyp_ver is not compatible with installed version: $hyp_ver"
      trace debug "The new guest will not be compatible with current host software."
      trace debug "Host upgrade is required"
    fi
    
    # Store the packagename for later
    mkdir -p ${upgrade_dir}
    rm -rf  ${upgrade_dir}/*
    basename $pkgfile > ${upgrade_dir}/pkg_name

    install_hypervisor_pkg $pkgfile $inst_dir/$vimg_pkg_gz

    # Make sure the next boot device is configured
    # as the current device from which the system 
    # booted.
    set_boot_from_same_disk
  fi

  # done, cleanup install dir
  rm -fr $inst_dir
  rm -fr $pkgfile
} 

rollback_package() {
  rollback_img $1
  rollback_hypervisor_pkg
}

compat_check() {
  local pkgfile=$1
  local min_ver=`tar -Oxzf $pkgfile | grep 'MINIMUM-VERSION' | sed 's/^.*=//'`

  if [ -f /etc/compatible_version ]; then
    cur_ver=`cat /etc/compatible_version`
  else
    cur_ver=`cat /etc/jrelease`
  fi

  if is_version_compatible $cur_ver $min_ver; then
    echo "Host OS compatible"
  else 
    echo "Host OS incompatible"
  fi 
}

case "$op" in
  add)
  if [ ! -f "$vj_pkg" ]; then
    error "Package $vj_pkg is not found, abort!"
  fi
  install_package $vj_pkg $opt_slot
  ;;

  rollback)
  rollback_package $opt_slot
  ;;

  compat-check)
  if [ ! -f "$vj_pkg" ]; then
    error "Package $vj_pkg is not found, abort!"
  fi 
  compat_check $vj_pkg
  ;;

  *)
  error "Unknown command $op"
esac
